<?php
  
namespace App\Http\Controllers;
  
use Illuminate\Http\Request;
use Illuminate\View\View;
use App\Models\JobBoard;


use App\Jobs\SendMessage;
use App\Models\Message;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use App\Helpers\Bbcode;
use Illuminate\Container\Attributes\DB;
use Google\Client;
use Google\Service\Calendar;


class HomeController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }
  
    /**
     * Show the application dashboard.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    // public function index(): View
    // {
    //     return view('home');
    // } 
    // app/Http/Controllers/DashboardController.php


// public function index()
// {
//     $user = auth()->user();

//     // Example relationships – adjust to your actual schema
//     // assumed pivot table like job_user or applications
//     $appliedJobs = $user->appliedJobs()
//         ->withPivot(['status', 'created_at'])
//         ->latest('pivot_created_at')
//         ->get();

//     // If you don’t have saved jobs yet, just use empty collection
//     $savedJobs = method_exists($user, 'savedJobs')
//         ? $user->savedJobs()->get()
//         : collect();

//     return view('home', compact('appliedJobs', 'savedJobs'));
// }

// app/Http/Controllers/DashboardController.php

public function index()
{
    $user = auth()->user();

    // Applied jobs + questions
    $appliedJobs = $user->appliedJobs()
        ->with(['questions'])
        ->withPivot(['status', 'created_at'])
        ->latest('pivot_created_at')
        ->get();

    // Saved jobs + questions
    $savedJobs = method_exists($user, 'savedJobs')
        ? $user->savedJobs()->with('questions')->get()
        : collect();

    // IDs of applied jobs
    $appliedIds = $appliedJobs->pluck('id');

    // Recommended jobs sorted by ad tier
    $recommendedJobs = JobBoard::with('questions')
        ->whereNotIn('id', $appliedIds)
        ->orderByRaw("
            CASE 
                WHEN ad_type = 'platinum' THEN 1
                WHEN ad_type = 'premium' THEN 2
                ELSE 3
            END
        ")
        ->orderBy('created_at', 'desc') // newest inside each tier
        ->take(6)
        ->get();

    return view('home', [
        'appliedJobs'     => $appliedJobs,
        'savedJobs'       => $savedJobs,
        'recommendedJobs' => $recommendedJobs,
    ]);
}




  
    /**
     * Show the application dashboard.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */


public function adminHome(): View
{
    $companyId = auth()->user()->company?->id;

    $jobCounts = [
        'active'  => JobBoard::where('company_id', $companyId)->where('status', 'active')->count(),
        'open'    => JobBoard::where('company_id', $companyId)->where('status', 'open')->count(),
        'expired' => JobBoard::where('company_id', $companyId)->where('status', 'expired')->count(),
        'closed'  => JobBoard::where('company_id', $companyId)->where('status', 'closed')->count(),
    ];

    // Fetch all jobs grouped by status + questions
    $activeJobs = JobBoard::with('questions')
        ->where('company_id', $companyId)
        ->where('status', 'active')
        ->get();

    $openJobs = JobBoard::with('questions')
        ->where('company_id', $companyId)
        ->where('status', 'open')
        ->get();

    $expiredJobs = JobBoard::with('questions')
        ->where('company_id', $companyId)
        ->where('status', 'expired')
        ->get();

    $closedJobs = JobBoard::with('questions')
        ->where('company_id', $companyId)
        ->where('status', 'closed')
        ->get();

    // Convert BBCode → HTML
    foreach ([$activeJobs, $openJobs, $expiredJobs, $closedJobs] as $jobs) {
        foreach ($jobs as $job) {
            $job->description_html = Bbcode::toHtml($job->description);
            $job->summary_html     = Bbcode::toHtml($job->summary);
        }
    }

    return view('adminHome', compact('jobCounts', 'activeJobs', 'openJobs', 'expiredJobs', 'closedJobs'));
}



  
    /**
     * Show the application dashboard.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    public function managerHome(): View
    {
        return view('managerHome');
    }
  public function messages(): JsonResponse {
    return response()->json(Message::with('user')->get());
}




// public function message(Request $request): JsonResponse {
//     $message = Message::create([
//         'user_id' => auth()->id(),
//         'text' => $request->get('text'),
//     ]);
//     SendMessage::dispatch($message);

//     return response()->json(['success' => true]);
// }



public function uploadResume(Request $request)
{
    $request->validate([
        'resume' => 'required|mimes:pdf,doc,docx|max:2048',
    ]);

    $path = $request->file('resume')->store('resumes', 'public');

    $user = auth()->user();
    $user->resume_path = $path;
    $user->save();

    return back()->with('success', 'Resume uploaded successfully.');
}

public function uploadCoverLetter(Request $request)
{
    $request->validate([
        'cover_letter' => 'required|mimes:pdf,doc,docx|max:2048',
    ]);

    $path = $request->file('cover_letter')->store('cover_letters', 'public');

    $user = auth()->user();
    $user->cover_letter_path = $path;
    $user->save();

    return back()->with('success', 'Cover letter uploaded successfully.');
}



public function save(Request $request)
{
    try {
        $request->validate([
            'user_id' => 'required|integer|exists:users,id',
        ]);

        // Check if already saved
        $exists = \DB::table('employer_directory')
            ->where('employer_id', auth()->id())
            ->where('candidate_id', $request->user_id)
            ->exists();

        if ($exists) {
            return response()->json([
                'success' => false,
                'message' => 'Candidate is already in your directory.'
            ]);
        }

        // Save to directory
        \DB::table('employer_directory')->insert([
            'employer_id'  => auth()->id(),
            'candidate_id' => $request->user_id,
            'created_at'   => now(),
            'updated_at'   => now(),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Candidate saved to your directory.'
        ]);

    } catch (\Exception $e) {

        return response()->json([
            'success' => false,
            'message' => 'An error occurred while saving.',
            'error'   => $e->getMessage()
        ]);
    }
}


public function list()
{
    try {

        $saved = \DB::table('employer_directory')
            ->join('users', 'users.id', '=', 'employer_directory.candidate_id')
            ->join('applications', function($join) {
                $join->on('applications.user_id', '=', 'users.id')
                     ->whereRaw('applications.id = (
                         SELECT id FROM applications 
                         WHERE applications.user_id = users.id 
                         ORDER BY id DESC LIMIT 1
                     )');
            })
            ->join('job_boards', 'job_boards.id', '=', 'applications.job_board_id')
            ->where('employer_directory.employer_id', auth()->id())
            ->select(
                'users.name',
                'job_boards.location',
                'job_boards.category',
                'applications.id as application_id'
            )
            ->get();

        return response()->json([
            'success' => true,
            'results' => $saved
        ]);

    } catch (\Exception $e) {

        // Return the exact SQL error
        return response()->json([
            'success' => false,
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);
    }
}


public function recommended()
{
    try {
        $matches = \DB::table('applications')
            ->join('users', 'users.id', '=', 'applications.user_id')
            ->join('job_boards', 'job_boards.id', '=', 'applications.job_board_id')
            ->select(
                'users.name',
                'job_boards.location',
                'job_boards.category',
                'applications.id as application_id'
            )
            ->orderBy('applications.id', 'desc')
            ->limit(10)
            ->get();

        return response()->json([
            'success' => true,
            'results' => $matches
        ]);

    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
}



public function jobPerformance()
{
    try {

        // 1. Base job performance (your working query)
        $jobs = \DB::table('job_boards')
            ->leftJoin('applications', 'applications.job_board_id', '=', 'job_boards.id')
            ->select(
                'job_boards.id',
                'job_boards.job_title',
                'job_boards.location',
                'job_boards.category',

                \DB::raw('COUNT(applications.id) as applications'),
                \DB::raw('SUM(applications.status = "Shortlisted") as shortlisted'),
                \DB::raw('SUM(applications.status = "Pending") as pending'),
                \DB::raw('SUM(applications.status = "Rejected") as rejected'),
                \DB::raw('SUM(applications.status = "Interview") as interview')
            )
            ->groupBy(
                'job_boards.id',
                'job_boards.job_title',
                'job_boards.location',
                'job_boards.category'
            )
            ->orderBy('job_boards.id', 'desc')
            ->get();


        // 2. Add weekly analytics, trend arrows, and AI insights
        foreach ($jobs as $job) {

            // Weekly application counts
            $weekly = \DB::table('applications')
                ->select(
                    \DB::raw('COUNT(*) as cnt'),
                    \DB::raw('WEEK(created_at) as week')
                )
                ->where('job_board_id', $job->id)
                ->groupBy('week')
                ->orderBy('week')
                ->get();

            $job->weekly_applications = $weekly->pluck('cnt');
            $job->weeks = $weekly->map(fn($w) => "Week " . $w->week);

            // Last week count
            $job->last_week = \DB::table('applications')
                ->where('job_board_id', $job->id)
                ->whereRaw('WEEK(created_at) = WEEK(NOW()) - 1')
                ->count();

            // Trend arrow logic
            $job->trend = $job->applications > $job->last_week
                ? "up"
                : ($job->applications < $job->last_week ? "down" : "equal");

            // AI Insight
            if ($job->rejected > $job->shortlisted) {
                $job->ai_insight = "This job is underperforming — too many rejections compared to shortlisted candidates.";
            } elseif ($job->pending > 10) {
                $job->ai_insight = "High pending count — consider reviewing applications faster.";
            } elseif ($job->shortlisted > ($job->applications * 0.5)) {
                $job->ai_insight = "Strong candidate pool — this job is attracting high‑quality applicants.";
            } else {
                $job->ai_insight = "Performance is stable. No major issues detected.";
            }
        }


        return response()->json([
            'success' => true,
            'results' => $jobs
        ]);

    } catch (\Exception $e) {

        return response()->json([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
}


public function jobPerformanceTracking()
{
    $jobs = \DB::table('job_boards')
        ->select('id', 'job_title', 'location', 'category', 'views')
        ->orderBy('views', 'desc')
        ->get();

    return response()->json([
        'success' => true,
        'results' => $jobs
    ]);
}



public function profileSettings()
{
    $user = auth()->user();

    return response()->json([
        'success' => true,
        'user' => auth()->user(),
        'company' => $user->company
    ]);
}

public function updateProfileSettings(Request $request)
{
    $user = auth()->user();
    $company = $user->company; // your hasOne relationship

    // Update USER
    $user->update([
        'name'     => $request->name,
        'email'    => $request->email,
        'phone'    => $request->phone,
        'location' => $request->location,
    ]);

    // Update COMPANY
    $company->update([
        // 'name'        => $request->company_name,
        'industry'    => $request->industry,
        'website'     => $request->website,
        'address'     => $request->address,
        'phone'       => $request->company_phone,
        'state'       => $request->state,
        'postcode'    => $request->postcode,
        'description' => $request->description,
    ]);

    // Handle LOGO upload
    if ($request->hasFile('logo')) {
        $path = $request->file('logo')->store('company_logos', 'public');
        $company->logo_url = "/storage/" . $path;
        $company->save();
    }

    return response()->json(['success' => true]);
}

public function transactions()
{
    $user = auth()->user();

    // Get the company that belongs to the user
    $company = $user->company;

    if (!$company) {
        return response()->json([
            'success' => true,
            'transactions' => []
        ]);
    }

    // Get all job board transactions for this company
    $transactions = $company->jobBoards()
        ->orderBy('id', 'asc')
        ->get([
            'id',
            'ad_type',
            'job_title',
            'location',
            'created_at as transaction_date',
            'ad_rev_status',
            'ad_price',
            'ad_duration',
            'payment_intent_id'
        ]);

    return response()->json([
        'success' => true,
        'transactions' => $transactions
    ]);
}



public function transactionReports(Request $request)
{
    $company = auth()->user()->company;

    if (!$company) {
        return response()->json([
            'success' => true,
            'years' => [],
            'months' => [],
            'transactions' => [],
        ]);
    }

    // Get all transactions
    $transactions = $company->jobBoards()
        ->orderBy('created_at', 'desc')
        ->get([
            'id',
            'ad_type',
            'job_title',
            'location',
            'created_at as transaction_date',
            'ad_rev_status',
            'ad_price',
            'ad_duration',
            'payment_intent_id'
        ]);

    // Extract all years available
    $years = $transactions
        ->pluck('transaction_date')
        ->map(fn($d) => \Carbon\Carbon::parse($d)->year)
        ->unique()
        ->sort()
        ->values();

    // Extract all months available (1–12)
    $months = $transactions
        ->pluck('transaction_date')
        ->map(fn($d) => \Carbon\Carbon::parse($d)->month)
        ->unique()
        ->sort()
        ->values();

    // Apply filters if provided
    $yearFilter = $request->query('year');
    $monthFilter = $request->query('month');

    $filtered = $transactions;

    if ($yearFilter) {
        $filtered = $filtered->filter(function ($t) use ($yearFilter) {
            return \Carbon\Carbon::parse($t->transaction_date)->year == $yearFilter;
        });
    }

    if ($monthFilter) {
        $filtered = $filtered->filter(function ($t) use ($monthFilter) {
            return \Carbon\Carbon::parse($t->transaction_date)->month == $monthFilter;
        });
    }

    // Calculate total spent
    $totalSpent = round($filtered->sum('ad_price'), 2);

    return response()->json([
        'success' => true,
        'years' => $years,
        'months' => $months,
        'total_spent' => $totalSpent,
        'count' => $filtered->count(),
        'transactions' => $filtered->values(),
    ]);
}


public function exportTransactionReports(Request $request)
{
    $format = $request->query('format', 'csv');
    $company = auth()->user()->company;

    $transactions = $company->jobBoards()
        ->orderBy('created_at', 'desc')
        ->get([
            'ad_type',
            'job_title',
            'location',
            'created_at as transaction_date',
            'ad_rev_status',
            'ad_price',
            'ad_duration',
            'payment_intent_id'
        ]);

    if ($format === 'csv') {
        $filename = 'transactions.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"$filename\"",
        ];

        $callback = function () use ($transactions) {
            $out = fopen('php://output', 'w');
            fputcsv($out, ['Ad Type','Job Title','Location','Transaction Date','Status','Price','Duration','Payment Intent ID']);

            foreach ($transactions as $t) {
                fputcsv($out, [
                    $t->ad_type,
                    $t->job_title,
                    $t->location,
                    $t->transaction_date,
                    $t->ad_rev_status,
                    $t->ad_price,
                    $t->ad_duration,
                    $t->payment_intent_id,
                ]);
            }

            fclose($out);
        };

        return response()->stream($callback, 200, $headers);
    }

    if ($format === 'pdf') {
        // Use your preferred PDF library (e.g. barryvdh/laravel-dompdf)
        // Example:
        // $pdf = PDF::loadView('reports.transactions-pdf', compact('transactions'));
        // return $pdf->download('transactions.pdf');

        // Placeholder:
        return response('PDF generation not yet implemented', 501);
    }

    return response('Invalid format', 400);
}



public function jobTrendsData(Request $request)
{
    $job = $request->query('job');
    $location = $request->query('location');

    $query = JobBoard::query();

    if ($job) {
        $query->where('job_title', $job);
    }

    if ($location) {
        $query->where('location', 'LIKE', "%{$location}%");
    }

    $results = $query
        ->selectRaw('job_title, COUNT(*) as applicants')
        ->groupBy('job_title')
        ->orderBy('job_title')
        ->get();

    return response()->json([
        'success' => true,
        'data' => $results
    ]);
}



// Google Meet Functions

public function redirect()
{
    $client = new Google\Client();
    $client->setAuthConfig(storage_path('app/google-oauth.json'));
    $client->addScope(Google\Service\Calendar::CALENDAR);
    $client->setRedirectUri(url('/google/callback'));

    return redirect()->away($client->createAuthUrl());
}

public function callback(Request $request)
{
    $client = new Google\Client();
    $client->setAuthConfig(storage_path('app/google-oauth.json'));
    $client->setRedirectUri(url('/google/callback'));

    $token = $client->fetchAccessTokenWithAuthCode($request->code);

    session(['google_token' => $token]);

    return redirect('/dashboard');
}














    
}