<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Google\Client;
use Google\Service\Calendar;
use Google\Service\Calendar\Event;
use Google\Service\Calendar\EventDateTime;
use Carbon\Carbon;
use App\Mail\InterviewScheduledMail;
use Illuminate\Support\Facades\Mail;
use App\Models\Application;



class GoogleController extends Controller
{
    private function googleClient()
    {
        $client = new Client();
        $client->setClientId(config('google.client_id'));
        $client->setClientSecret(config('google.client_secret'));
        $client->setRedirectUri(config('google.redirect_uri'));
        $client->setAccessType('offline');
        $client->setPrompt('consent');
        $client->setScopes(config('google.scopes'));

        return $client;
    }

    public function redirectToGoogle()
    {
        $client = $this->googleClient();
        return redirect()->away($client->createAuthUrl());
    }

    // public function handleGoogleCallback(Request $request)
    // {
    //     $client = $this->googleClient();

    //     if ($request->has('code')) {
    //         $token = $client->fetchAccessTokenWithAuthCode($request->code);

    //         dd($token);
    //         dd(auth()->user()->id);

    //         auth()->user()->update([
    //             'google_token' => json_encode($token)
    //         ]);

    //         return redirect('/admin/home')->with('success', 'Google connected successfully');
    //     }

    //     return redirect('/admin/home')->with('error', 'Google authentication failed');
    // }
// public function handleGoogleCallback(Request $request)
// {
//     dd([
//         'session_id' => session()->getId(),
//         'cookies' => $_COOKIE,
//         'user' => auth()->user(),
//         'has_code' => $request->has('code'),
//         'full_request' => $request->all()
//     ]);
// }

public function handleGoogleCallback(Request $request)
{
    $client = $this->googleClient();

    if (!$request->has('code')) {
        return redirect('/admin/home')->with('error', 'Google authentication failed');
    }

    $token = $client->fetchAccessTokenWithAuthCode($request->code);

    if (isset($token['error'])) {
        return redirect('/admin/home')->with('error', 'Google OAuth Error: ' . $token['error']);
    }

    if (!auth()->check()) {
        return redirect('/login')->with('error', 'You must be logged in before connecting Google.');
    }

    auth()->user()->update([
        'google_token' => json_encode($token)
    ]);

    return redirect('/admin/home');
    // \log::info('Google OAuth Token', ['token' => $token]);
}







public function send(Request $request)
{
    try {
        $user = auth()->user();

        if (!$user->google_token) {
            return response()->json([
                'error' => 'Google account not connected. Please connect Google first.'
            ], 400);
        }

        // Restore Google token
        $token = json_decode($user->google_token, true);

        $client = $this->googleClient();
        $client->setAccessToken($token);

        // Refresh if expired
        if ($client->isAccessTokenExpired()) {
            $client->fetchAccessTokenWithRefreshToken($client->getRefreshToken());
            $user->update(['google_token' => json_encode($client->getAccessToken())]);
        }

        $service = new Calendar($client);

        // Create Google Calendar Event
$event = new Event([
    'summary' => $request->title,
    'description' => $request->description,
    'start' => new EventDateTime([
        'dateTime' => Carbon::parse($request->start)->toRfc3339String(),
        'timeZone' => 'Australia/Brisbane',
    ]),
    'end' => new EventDateTime([
        'dateTime' => Carbon::parse($request->end)->toRfc3339String(),
        'timeZone' => 'Australia/Brisbane',
    ]),
    'conferenceData' => [
        'createRequest' => [
            'requestId' => uniqid(),
            'conferenceSolutionKey' => ['type' => 'hangoutsMeet']
        ]
    ]
]);


        $createdEvent = $service->events->insert(
            'primary',
            $event,
            ['conferenceDataVersion' => 1]
        );

        $meetUrl = $createdEvent->hangoutLink;

        // -----------------------------
        // PROCESS RECIPIENTS
        // -----------------------------

        $toRecipients  = $request->to ?? [];
        $ccRecipients  = $request->cc ?? [];
        $bccRecipients = $request->bcc ?? [];
        $description   = $request->description ?? '';

        foreach ($toRecipients as $recipient) {

            // Find application by email
       $application = Application::whereHas('user', function ($q) use ($recipient) {
    $q->where('email', $recipient['email']);
})->first();


            if ($application) {
                // Update application record
                $application->interview_date = Carbon::parse($request->start)->format('Y-m-d H:i:s');
                $application->interview_link = $meetUrl;
                $application->status = 'interview_scheduled';
                $application->save();
            }

            // Format date/time for email
                $start = Carbon::parse($request->start);
            $end   = Carbon::parse($request->end);
            $date = $start->format('l, d F Y');
            $startTime = $start->format('g:i A');
            $endTime   = $end->format('g:i A');
            $startTimeUtc = $start->copy()->utc()->format('Ymd\THis\Z');
            $endTimeUtc   = $end->copy()->utc()->format('Ymd\THis\Z');



            // Send email
                Mail::to($recipient['email'])
                    ->cc($ccRecipients)
                    ->bcc($bccRecipients)
                    ->send(new InterviewScheduledMail(
                        applicant: $recipient['name'],
                        sender: auth()->user()->name,
                        date: $date,
                        startTime: $startTime,
                        endTime: $endTime,
                        meetUrl: $meetUrl,
                        description: $description,
                        startTimeUtc: $startTimeUtc,
                        endTimeUtc: $endTimeUtc
                    ));


        }

        return response()->json([
            'meet_url' => $meetUrl,
            'event_id' => $createdEvent->id
        ]);

    } catch (\Exception $e) {
        return response()->json([
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ], 500);
    }
}

}





// public function scheduleInterview(Request $request)
// {
//     try {
//         $user = auth()->user();

//         if (!$user->google_token) {
//             return response()->json([
//                 'error' => 'Google account not connected. Please connect Google first.'
//             ], 400);
//         }

//         $token = json_decode($user->google_token, true);

//         $client = $this->googleClient();
//         $client->setAccessToken($token);

//         if ($client->isAccessTokenExpired()) {
//             $client->fetchAccessTokenWithRefreshToken($client->getRefreshToken());
//             $user->update(['google_token' => json_encode($client->getAccessToken())]);
//         }

//         $service = new Calendar($client);

//         $event = new Event([
//             'summary' => $request->title,
//             'description' => $request->description,
//             'start' => new EventDateTime([
//                 'dateTime' => $request->start_time,
//                 'timeZone' => 'Australia/Brisbane',
//             ]),
//             'end' => new EventDateTime([
//                 'dateTime' => $request->end_time,
//                 'timeZone' => 'Australia/Brisbane',
//             ]),
//             'conferenceData' => [
//                 'createRequest' => [
//                     'requestId' => uniqid(),
//                     'conferenceSolutionKey' => ['type' => 'hangoutsMeet']
//                 ]
//             ]
//         ]);

//         $createdEvent = $service->events->insert(
//             'primary',
//             $event,
//             ['conferenceDataVersion' => 1]
//         );

//         //  UPDATE APPLICATION RECORD
//      $application = \App\Models\Application::find($request->application_id);

// if (!$application) {
//     return response()->json([
//         'error' => 'Application not found. ID received: ' . $request->application_id
//     ], 404);
// }

// $application->interview_date = Carbon::parse($request->start_time)->format('Y-m-d H:i:s');
// $application->interview_link = $createdEvent->hangoutLink;
// $application->status = 'interview_scheduled';
// $application->save();



// // SEND EMAIL TO APPLICANT

// $datetime = \Carbon\Carbon::parse($application->interview_date);

// $date = $datetime->format('l, d F Y');   // Monday, 26 January 2026
// $time = $datetime->format('g:i A');      // 10:30 AM

// Mail::to($application->user->email)->send(
//     new InterviewScheduledMail(
//         $application->user->name,   // applicant name
//         $date,                      // formatted date
//         $time,                      // formatted time
//         $createdEvent->hangoutLink, // Google Meet link
//         auth()->user()->name        // sender name
//     )
// );





//         return response()->json([
//             'meet_url' => $createdEvent->hangoutLink,
//             'event_id' => $createdEvent->id
//         ]);

//     } catch (\Exception $e) {
//         return response()->json([
//             'error' => $e->getMessage(),
//             'trace' => $e->getTraceAsString()
//         ], 500);
//     }
// }

// }
