<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Company;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class CompanyController extends Controller
{
    public function create()
    {
        return view('auth.company-register');
    }

    /**
     * AJAX endpoint to lookup ABN details
     */
public function lookup(Request $request)
{
    $request->validate([
        'abn' => 'required|string',
    ]);

    $guid = config('services.abn.guid');
    $url  = "https://abr.business.gov.au/json/AbnDetails.aspx?abn={$request->abn}&guid={$guid}";

    try {
        $response = Http::get($url);

        if (!$response->ok()) {
            return response()->json([
                'error'  => 'Failed to fetch ABN details',
                'status' => $response->status(),
                'body'   => $response->body(),
            ], $response->status());
        }

        // Get raw body (JSONP format)
        $raw = $response->body();

        // Remove "callback(" prefix and trailing ")"
        $clean = preg_replace('/^callback\((.*)\)$/', '$1', $raw);

        // Decode JSON
        $data = json_decode($clean, true);

        if (empty($data['Abn'])) {
            return response()->json([
                'error' => 'No company found for this ABN',
                'raw'   => $data,
            ], 404);
        }

        return response()->json([
            'abn'      => $data['Abn'],
            'name'     => $data['EntityName'] ?? null,
            'status'   => $data['AbnStatus'] ?? null,
            'type'     => $data['EntityTypeName'] ?? null,
            'state'    => $data['AddressState'] ?? null,
            'postcode' => $data['AddressPostcode'] ?? null,
        ]);

    } catch (\Exception $e) {
        return response()->json([
            'error'   => 'Exception occurred while fetching ABN details',
            'message' => $e->getMessage(),
        ], 500);
    }
}



    /**
     * Store company after ABN lookup
     */
    // public function store(Request $request)
    // {
    //     $validated = $request->validate([
    //         'abn'      => 'required|string',
    //         'name'     => 'nullable|string',
    //         'status'   => 'nullable|string',
    //         'type'     => 'nullable|string',
    //         'state'    => 'nullable|string',
    //         'postcode' => 'nullable|string',
    //     ]);

    //     // Generate UUID once
    //     $uuid = Str::uuid()->toString();

    //     $company = Company::create(array_merge($validated, [
    //         'c_uuid'  => $uuid,
    //         'user_id' => null, // leave empty until user exists
    //     ]));

    //     // Flash UUID into session
    //     session()->put('company_uuid', $uuid);

    //     return redirect()
    //         ->route('register')
    //         ->with('success', 'Company registered successfully!');
    // }

    public function store(Request $request)
{
    $validated = $request->validate([
        'abn'      => 'required|string',
        'name'     => 'nullable|string',
        'status'   => 'nullable|string',
        'type'     => 'nullable|string',
        'state'    => 'nullable|string',
        'postcode' => 'nullable|string',
    ]);

    try {
        $uuid = Str::uuid()->toString();

        Company::create(array_merge($validated, [
            'c_uuid'  => $uuid,
            'user_id' => null,
        ]));

        session()->put('company_uuid', $uuid);

        return redirect()
            ->route('register')
            ->with('success', 'Company registered successfully!');
    } 
    catch (\Illuminate\Database\QueryException $e) {

        // Extract SQL error code
        $errorCode = $e->errorInfo[1] ?? null;

        // 1062 = Duplicate entry
        if ($errorCode == 1062) {
            return redirect()
                ->back()
                ->with('error', 'Company already exists. Please log in below.')
                ->with('show_login', true)
                ->withInput();
        }

        // Any other DB error → show exact message
        return redirect()
            ->back()
            ->with('error', $e->getMessage())
            ->withInput();
    }
}


    public function show(Company $company)
    {
        return view('companies.show', compact('company'));
    }
}
